#region Using statements

using System;
using System.Data;
using System.Data.SqlClient;
using System.Text.RegularExpressions;

using Method = System.Reflection.MethodBase;

using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.ExceptionManagement;

using BLOODUNITFINANCE = gov.va.med.vbecs.Common.VbecsTables.BloodUnitFinance;
using BUSH_TABLE = gov.va.med.vbecs.Common.VbecsTables.BloodUnitShipment;

#endregion

namespace gov.va.med.vbecs.BOL
{

	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Luke Meyer</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>10/4/2002</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>Contains financial information about a blood unit</summary>

	#endregion
	public class BloodUnitFinance: BaseBusinessObject
	{
		#region Variables

		/// <summary>
		/// Where the class information is stored
		/// </summary>
		private System.Data.DataRow _drBloodUnitFinance;

		/// <summary>
		/// Table used to create the datarow
		/// </summary>
		private System.Data.DataTable _bloodUnitFinanceDataTable;

		#endregion

		#region Constructors

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5224"> 
		///		<ExpectedInput>Valid GUID</ExpectedInput>
		///		<ExpectedOutput>BloodUnitFinance object initialized</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5225"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		///  BloodUnitFinance Constructor
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		public BloodUnitFinance(System.Guid bloodUnitGuid)
		{
			_bloodUnitFinanceDataTable = DAL.BloodUnitFinance.GetBloodUnitFinanceByUnitGuid(bloodUnitGuid);

			//Do we need to make a new object?
			if (_bloodUnitFinanceDataTable.Rows.Count > 0)
			{
				_drBloodUnitFinance = _bloodUnitFinanceDataTable.Rows[0];
			}
			else
			{
				this.MakeNew(bloodUnitGuid);
			}

			_drBloodUnitFinance.AcceptChanges();
			_bloodUnitFinanceDataTable.ColumnChanged += new DataColumnChangeEventHandler(this.OnColumnChanged);
		}

		#endregion

		#region Events

		/// <summary>
		/// Sets the IsDirty flag so we don't need that line of code in EVERY property
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="eventArgs"></param>
		protected void OnColumnChanged(object sender, DataColumnChangeEventArgs eventArgs)
		{
			this.IsDirty = true;
		}

		#endregion

		#region Methods

		/// <summary>
		/// MakeNew
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		private void MakeNew(System.Guid bloodUnitGuid)
		{
			if (_bloodUnitFinanceDataTable == null)
			{
				throw new NullReferenceException("Unit Finance DataTable is null");
			}

			this._drBloodUnitFinance = _bloodUnitFinanceDataTable.NewRow();

			this._drBloodUnitFinance[BLOODUNITFINANCE.BloodUnitFinanceGuid] = System.Guid.NewGuid();
			this._drBloodUnitFinance[BLOODUNITFINANCE.BloodUnitGuid] = bloodUnitGuid;
			this._drBloodUnitFinance[BLOODUNITFINANCE.UnitCost] = System.Decimal.MinValue;
			this._drBloodUnitFinance[BLOODUNITFINANCE.SpecialTestingCost] = System.Decimal.MinValue;
			this._drBloodUnitFinance[BLOODUNITFINANCE.ReturnCreditAmount] = System.Decimal.MinValue;
			this._drBloodUnitFinance[BUSH_TABLE.OutgoingShipmentGuid] = Guid.Empty;
			this._drBloodUnitFinance[BLOODUNITFINANCE.LastUpdateUser] = Common.LogonUser.LogonUserName;

			this._bloodUnitFinanceDataTable.Rows.Add(this._drBloodUnitFinance);

			this.IsNew = true;
		}


		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5226"> 
		///		<ExpectedInput>productshippers, localsuppierguid</ExpectedInput>
		///		<ExpectedOutput>UnitCost</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5227"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// LoadDefaultCostInformation
		/// </summary>
		/// <param name="productShippers"></param>
		/// <param name="localSupplierGuid"></param>
		public void LoadDefaultCostInformation(BOL.ProductShipperCollection productShippers, System.Guid localSupplierGuid)
		{
			try
			{
				int val = 1;

				BOL.ProductShipper ps = productShippers.GetElementByLocalSupplierGuid(localSupplierGuid);

				if (ps == null)
				{
					val = 0;
				}

				switch(val)
				{
					case 0:
						break;

					case 1:
						this.UnitCost = ps.ProductCost;
						break;

					default:
						break;
				}
			}
			catch (Exception err)
			{
				throw new BOL.BusinessObjectException("Error Loading Defualt Information", err);
			}
		}

		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5228"> 
		///		<ExpectedInput>unitid, productcode</ExpectedInput>
		///		<ExpectedOutput>Financial Data in unit</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5229"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///		
		/// <summary>
		/// LoadBloodUnitFinancialData
		/// </summary>
		public void LoadBloodUnitFinancialData(string unitID, string productCode)
		{
			if (this._drBloodUnitFinance == null)
			{
				DataRow drFinancial = DAL.BloodUnitFinance.GetBloodUnitFinancialData(unitID, productCode);
				this._drBloodUnitFinance = drFinancial;

				//Load financial object from data row
				this.UnitCost = drFinancial.IsNull(BLOODUNITFINANCE.UnitCost) ? 0 : (decimal)drFinancial[BLOODUNITFINANCE.UnitCost];
				this.SpecialTestingCost = drFinancial.IsNull(BLOODUNITFINANCE.SpecialTestingCost) ? 0 : (decimal)drFinancial[BLOODUNITFINANCE.SpecialTestingCost];
				this.ReturnCreditAmount = drFinancial.IsNull(BLOODUNITFINANCE.ReturnCreditAmount) ? 0 : (decimal)drFinancial[BLOODUNITFINANCE.ReturnCreditAmount];
			}
		}

		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5230"> 
		///		<ExpectedInput>UpdateFunction</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5231"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///		
		/// <summary>
		/// Save
		/// </summary>
		/// <param name="lastUpdateFunctionId">UC or calling method</param>
		public bool Save(Common.UpdateFunction lastUpdateFunctionId)
		{
			bool retVal = true;

			if ((this.IsNew) || (!this.IsNew && this.IsDirty))
			{
				System.Data.DataTable dtBloodUnitFinance = DAL.BloodUnitFinance.GetEmptyBloodUnitFinanceTableSchema(true);
				System.Data.DataRow drBloodUnitFinance = this.LoadDataRowFromThis(dtBloodUnitFinance.NewRow());
				dtBloodUnitFinance.Rows.Add(drBloodUnitFinance);

				dtBloodUnitFinance = Common.Utility.AppendLastUpdateInformation(dtBloodUnitFinance, lastUpdateFunctionId);

				retVal = DAL.BloodUnitFinance.Save(dtBloodUnitFinance.Rows[0]);

				//If the save was a success...
				if (retVal)
				{
					_drBloodUnitFinance.AcceptChanges();
				}
			}
			return(retVal);
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5232"> 
		///		<ExpectedInput>local supplier guid, product guid</ExpectedInput>
		///		<ExpectedOutput>decimal</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5233"> 
		///		<ExpectedInput>invalid guids</ExpectedInput>
		///		<ExpectedOutput>0</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Checks the ProductShipper table for ReturnCredit%.  If found, multiplies by BaseCost
		/// </summary>
		/// <param name="localSupplierGuid"></param>
		/// <param name="bloodProductGuid"></param>
		/// <returns></returns>
		public decimal CalculateReturnCreditAmount(System.Guid localSupplierGuid, System.Guid bloodProductGuid)
		{
			decimal returnCreditAmount = decimal.Zero;

			BOL.ProductShipperCollection tmpShippers = new BOL.ProductShipperCollection(bloodProductGuid);
			BOL.ProductShipper tmpPS = tmpShippers.GetElementByLocalSupplierGuid(localSupplierGuid);

			if (tmpPS != null)
			{
				returnCreditAmount = tmpPS.ReturnCreditPercentage / 100 * this.UnitCost;
			}

			return (returnCreditAmount);
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5234"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5235"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///		
		/// <summary>
		/// GetFinancialBloodUnits
		/// </summary>
		/// <returns></returns>
		public static DataTable GetFinancialBloodUnits()
		{
			return DAL.BloodUnitFinance.GetFinancialBloodUnits();
		}


		/// <summary>
		/// LoadFromDataRow
		/// </summary>
		/// <param name="dtRow"></param>
		protected override void LoadFromDataRow(System.Data.DataRow dtRow)
		{
			if (dtRow.Table.Columns.Contains(BLOODUNITFINANCE.BloodUnitFinanceGuid))
			{
				if (!dtRow.IsNull(BLOODUNITFINANCE.BloodUnitFinanceGuid))
				{
					this.BloodUnitFinanceGuid = (System.Guid)dtRow[BLOODUNITFINANCE.BloodUnitFinanceGuid];
				}
			}

			if (dtRow.Table.Columns.Contains(BLOODUNITFINANCE.BloodUnitGuid))
			{
				if (!dtRow.IsNull(BLOODUNITFINANCE.BloodUnitGuid))
				{
					this.BloodUnitGuid = (System.Guid)dtRow[BLOODUNITFINANCE.BloodUnitGuid];
				}
			}

			if (dtRow.Table.Columns.Contains(BUSH_TABLE.OutgoingShipmentGuid))
			{
				if (!dtRow.IsNull(BUSH_TABLE.OutgoingShipmentGuid))
				{
					this.OutgoingShipmentGuid = (System.Guid)dtRow[BUSH_TABLE.OutgoingShipmentGuid];
				}
			}

			if (dtRow.Table.Columns.Contains(BLOODUNITFINANCE.ReturnCreditAmount))
			{
				if (!dtRow.IsNull(BLOODUNITFINANCE.ReturnCreditAmount))
				{
					this.ReturnCreditAmount = (decimal)dtRow[BLOODUNITFINANCE.ReturnCreditAmount];
				}
			}

			if (dtRow.Table.Columns.Contains(BLOODUNITFINANCE.SpecialTestingCost))
			{
				if (!dtRow.IsNull(BLOODUNITFINANCE.SpecialTestingCost))
				{
					this.SpecialTestingCost = (decimal)dtRow[BLOODUNITFINANCE.SpecialTestingCost];
				}
			}

			if (dtRow.Table.Columns.Contains(BLOODUNITFINANCE.UnitCost))
			{
				if (!dtRow.IsNull(BLOODUNITFINANCE.UnitCost))
				{
					this.UnitCost = (decimal)dtRow[BLOODUNITFINANCE.UnitCost];
				}
			}

		}

		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5262"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5263"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// LoadDataRowFromThis
		/// </summary>
		/// <param name="dtRow"></param>
		/// <returns></returns>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dtRow)
		{
			try
			{
				//Dont populate DivisionCode in here -- BloodUnitFinance has no fields that need datetime conversion

				if (dtRow.Table.Columns.Contains(BLOODUNITFINANCE.BloodUnitFinanceGuid))
				{
					dtRow[BLOODUNITFINANCE.BloodUnitFinanceGuid] = this.BloodUnitFinanceGuid;
				}

				if (dtRow.Table.Columns.Contains(BLOODUNITFINANCE.BloodUnitGuid))
				{
					dtRow[BLOODUNITFINANCE.BloodUnitGuid] = this.BloodUnitGuid;
				}

				if (dtRow.Table.Columns.Contains(BUSH_TABLE.OutgoingShipmentGuid))
				{
					dtRow[BUSH_TABLE.OutgoingShipmentGuid] = this.OutgoingShipmentGuid;
				}

				if (dtRow.Table.Columns.Contains(BLOODUNITFINANCE.ReturnCreditAmount))
				{
					if (this.ReturnCreditAmount != decimal.MinValue)
					{
						dtRow[BLOODUNITFINANCE.ReturnCreditAmount] = this.ReturnCreditAmount;
					}
				}

				if (dtRow.Table.Columns.Contains(BLOODUNITFINANCE.SpecialTestingCost))
				{
					if (this.SpecialTestingCost != decimal.MinValue)
					{
						dtRow[BLOODUNITFINANCE.SpecialTestingCost] = this.SpecialTestingCost;
					}
				}

				if (dtRow.Table.Columns.Contains(BLOODUNITFINANCE.UnitCost))
				{
					if (this.UnitCost != decimal.MinValue)
					{
						dtRow[BLOODUNITFINANCE.UnitCost] = this.UnitCost;
					}
				}
				if(dtRow.Table.Columns.Contains(BLOODUNITFINANCE.LastUpdateUser))
				{
					dtRow[BLOODUNITFINANCE.LastUpdateUser] = Common.LogonUser.LogonUserName;
				}

				if(dtRow.Table.Columns.Contains(BLOODUNITFINANCE.RowVersion))
				{
					dtRow.Table.Columns[BLOODUNITFINANCE.RowVersion].ReadOnly = false;
					dtRow[BLOODUNITFINANCE.RowVersion] = this.RowVersion;
				}

				return dtRow;
			}
			catch (Exception err)
			{
				throw new BOL.BusinessObjectException("Invalid Data", err);
			}
		}

		#endregion

		#region Properties

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5236"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Decimal</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5237"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// MaximumBaseCostValue
		/// </summary>
		public static decimal MaximumBaseCostValue
		{
			get
			{
				return (decimal)99999;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5238"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>decimal</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5239"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// MaximumSpecialTestCost
		/// </summary>
		public static decimal MaximumSpecialTestCost
		{
			get
			{
				return (decimal)999;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5240"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>decimal</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5241"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// MaximumReturnCreditAmount
		/// </summary>
		public static decimal MaximumReturnCreditAmount
		{
			get
			{
				return (decimal)99999;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5242"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5243"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BloodUnitFinanceGUID
		/// </summary>
		public System.Guid BloodUnitFinanceGuid
		{
			get
			{
				if (this._drBloodUnitFinance.IsNull(BLOODUNITFINANCE.BloodUnitFinanceGuid))
				{
					return System.Guid.Empty;
				}

				return (System.Guid)this._drBloodUnitFinance[BLOODUNITFINANCE.BloodUnitFinanceGuid];
			}
			set
			{
				this._drBloodUnitFinance[BLOODUNITFINANCE.BloodUnitFinanceGuid] = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5244"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5245"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// BloodUnitGuid
		/// </summary>
		public System.Guid BloodUnitGuid
		{
			get
			{
				if (this._drBloodUnitFinance.IsNull(BLOODUNITFINANCE.BloodUnitGuid))
				{
					return System.Guid.Empty;
				}
				return (System.Guid)this._drBloodUnitFinance[BLOODUNITFINANCE.BloodUnitGuid];
			}
			set
			{
				this._drBloodUnitFinance[BLOODUNITFINANCE.BloodUnitGuid] = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5246"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5247"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// OutgoingShipmentGuid
		/// </summary>
		public System.Guid OutgoingShipmentGuid
		{
			get
			{
				if (_drBloodUnitFinance.IsNull(BUSH_TABLE.OutgoingShipmentGuid))
				{
					return System.Guid.Empty;
				}
				return (System.Guid)_drBloodUnitFinance[BUSH_TABLE.OutgoingShipmentGuid];
			}
			set
			{
				if (value == System.Guid.Empty)
				{
					_drBloodUnitFinance[BUSH_TABLE.OutgoingShipmentGuid] = DBNull.Value;
				}
				else
				{
					_drBloodUnitFinance[BUSH_TABLE.OutgoingShipmentGuid] = value;
				}
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5248"> 
		///		<ExpectedInput>decimal</ExpectedInput>
		///		<ExpectedOutput>decimal</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5249"> 
		///		<ExpectedInput>Invalid decimal</ExpectedInput>
		///		<ExpectedOutput>Broken Rule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// UnitCost
		/// </summary>
		public decimal UnitCost
		{
			get
			{
				if (this._drBloodUnitFinance.IsNull(BLOODUNITFINANCE.UnitCost))
				{
					return System.Decimal.MinValue;
				}
				return (System.Decimal)this._drBloodUnitFinance[BLOODUNITFINANCE.UnitCost];
			}
			set
			{
				RuleBroken(Method.GetCurrentMethod().Name, (value < 0) || (value > MaximumBaseCostValue));
				if (value != this.UnitCost)
					this.IsDirty = true;
				this._drBloodUnitFinance[BLOODUNITFINANCE.UnitCost] = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5250"> 
		///		<ExpectedInput>decimal</ExpectedInput>
		///		<ExpectedOutput>decimal</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5251"> 
		///		<ExpectedInput>Invalid decimal</ExpectedInput>
		///		<ExpectedOutput>BrokenRule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// SpecialTestingCost
		/// </summary>
		public decimal SpecialTestingCost
		{
			get
			{
				if (this._drBloodUnitFinance.IsNull(BLOODUNITFINANCE.SpecialTestingCost))
				{
					return System.Decimal.MinValue;
				}
				else
				{
					return (System.Decimal)this._drBloodUnitFinance[BLOODUNITFINANCE.SpecialTestingCost];
				}
			}
			set
			{
				RuleBroken(Method.GetCurrentMethod().Name, (value < 0) || (value > MaximumSpecialTestCost));
				if (value != this.SpecialTestingCost)
					this.IsDirty = true;
				this._drBloodUnitFinance[BLOODUNITFINANCE.SpecialTestingCost] = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5252"> 
		///		<ExpectedInput>decimal</ExpectedInput>
		///		<ExpectedOutput>decimal</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5253"> 
		///		<ExpectedInput>invalid decimal</ExpectedInput>
		///		<ExpectedOutput>Broken Rule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ReturnCreditAmount
		/// </summary>
		public decimal ReturnCreditAmount
		{
			get
			{
				if (this._drBloodUnitFinance.IsNull(BLOODUNITFINANCE.ReturnCreditAmount))
				{
					return System.Decimal.MinValue;
				}
				return (System.Decimal)this._drBloodUnitFinance[BLOODUNITFINANCE.ReturnCreditAmount];
			}
			set
			{
				RuleBroken(Method.GetCurrentMethod().Name, (value < 0) || (value > MaximumReturnCreditAmount));
				if (value != this.ReturnCreditAmount)
					this.IsDirty = true;
				this._drBloodUnitFinance[BLOODUNITFINANCE.ReturnCreditAmount] = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5254"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5255"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// IsReturnCreditAmountNull
		/// </summary>
		public bool IsReturnCreditAmountNull
		{
			get
			{
				return _drBloodUnitFinance.IsNull(BLOODUNITFINANCE.ReturnCreditAmount);
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5256"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5257"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// IsUnitCostNull
		/// </summary>
		public bool IsUnitCostNull
		{
			get
			{
				return _drBloodUnitFinance.IsNull(BLOODUNITFINANCE.UnitCost);
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5258"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5259"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// IsSpecialTestingCostNull
		/// </summary>
		public bool IsSpecialTestingCostNull
		{
			get
			{
				return _drBloodUnitFinance.IsNull(BLOODUNITFINANCE.SpecialTestingCost);
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/30/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5260"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>byte[]</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5261"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// RowVersion
		/// </summary>
		public new System.Byte[] RowVersion
		{
			get
			{
				if (_drBloodUnitFinance.IsNull(BLOODUNITFINANCE.RowVersion))
				{
					return null;
				}
				return (System.Byte[])_drBloodUnitFinance[BLOODUNITFINANCE.RowVersion];
			}
		}

		/// <summary>
		/// CR 3195
		/// </summary>
		public new System.DateTime LastUpdateDate  
		{
			get
			{
				if (_drBloodUnitFinance.IsNull(BLOODUNITFINANCE.LastUpdateDate)) 
				{
					return DateTime.MinValue;
				}
				return (System.DateTime)_drBloodUnitFinance[BLOODUNITFINANCE.LastUpdateDate];
			}
			set
			{
				_drBloodUnitFinance[BLOODUNITFINANCE.LastUpdateDate] = value;
			}
		}
		#endregion

	}
}